/*
 *  FlightSimu - Front End GUI to FlightGear / Moving Map Atlas
 *  Copyright (c) 2002, 2004 Roger Seguin <roger_seguin@msn.com>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

static char     RCSid[] =
    "$Id: flightsimu.C,v 1.18 2004/08/15 11:12:45 RogerSeguin Exp $";

#define COPYRIGHT	"Copyright (c) 2002, 2004 Roger Seguin "	\
    			"<roger_seguin@msn.com, http://rmlx.dyndns.org>"


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <memory.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <pthread.h>

#include <sys/time.h>
#include <sys/wait.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Protocols.h>
#include <Xm/RepType.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>


#include "flightgui.h"


#define APPNAME "FlightSimu"


/**********************************************************************/
namespace {
/**********************************************************************/
    const int       MAXCMDLINESIZE = 512;
    const int       MAXCODENAMESIZE = 32;
    const int       MAXDISPLAYSIZE = 64;
    const int       MAXLABELSIZE = 16;

    const char      DEFAULT_INI_FILENAME[] = ".flight.ini";
    const char      DEFAULT_HLP_FILENAME[] = ".flight.hlp";

    const char     *SIMUCMD_LABEL = "SIMULATOR:";

    const char     *FG_ROOT_LABEL = "FG_ROOT:";

    const char     *MAP_CMD_LABEL = "MAP_CMD:";
    const char     *ATLAS_DIR_LABEL = "ATLAS_DIR:";
    const char     *ATLAS_PORT_LABEL = "ATLAS_PORT:";
    const char     *ATLAS_XDISPLAY_LABEL = "ATLAS_XDISPLAY:";

    const char     *AIRCRAFT_LABEL = "AIRCRAFT:";
    const char     *AIRPORT_LABEL = "AIRPORT:";

    const char     *SEPARATOR_LABEL = "SEPARATOR:";

    const int       MAX_AIRCRAFTS = 100;
    const int       MAX_AIRPORTS = 500;

    char            m_acSimuCmd[MAXCMDLINESIZE] = "fgfs";

    char            m_acFG_Root[MAXCMDLINESIZE] = "/usr/lib/FlightGear";

    char            m_aacAircraftList[MAX_AIRCRAFTS][MAXCODENAMESIZE];
    char            m_aacAirportList[MAX_AIRPORTS][MAXCODENAMESIZE];

    int             m_iAircraft;
    int             m_iAirport;

    int             m_fHighVisibility = 0;
    int             m_fWindOff = 0;

    int             m_fUseAtlas = 1;
    char            m_acXDisplay[MAXDISPLAYSIZE];
    char            m_acAtlasXDisplay[MAXDISPLAYSIZE];
    char            m_acAtlasPort[8] = "5510";
    char            m_acMapCmd[MAXCODENAMESIZE] = "Atlas";
    char            m_acAtlasDir[MAXCODENAMESIZE] = "/opt/Atlas";

    struct TimeOffset {
	int             i;
	char            ac[4];
	                TimeOffset () {
	    i = 0, strcpy (ac, "0");
    }} m_oTimeOffset;

    int             m_fSimulatorRunning;
    int             m_fAtlasRunning;

    pid_t           m_iSimulatorPID;
    pid_t           m_iAtlasPID;

    XtAppContext    m_xAppContext;
    Display        *m_pxDisplay;
}


/**********************************************************************/
int System (const char *p_pcCmd, char *p_pcDisplay = 0,
	    pid_t * p_piPID = 0)
/**********************************************************************/
    /* MT-Safe version of system() - Supply the child process PID as soon
       as it is fork()'ed - Return the child completion status */
{
    const size_t    N = strlen (p_pcCmd) + 1;
    char            acFormat[16];
    char           *pcStr,	// 
                   *pcStr1,	// 
                   *pcStr2;
    char          **argv;
    int             argc;
    int             n;
    pid_t           pid;
    int             iChildStatus;

    pcStr = static_cast < char *>(malloc (N));
    pcStr1 = static_cast < char *>(malloc (N));
    pcStr2 = static_cast < char *>(malloc (N));
    argv = static_cast < char **>(malloc (N));
    if (!(pcStr && pcStr1 && pcStr2 && argv)) {
	perror ("malloc");
	return (-1);
    }
    sprintf (acFormat, "%%s %%%dc", N - 1);
    strcpy (pcStr, p_pcCmd);
    for (argc = 0;;) {
	memset (pcStr2, 0, N);
	n = sscanf (pcStr, acFormat, pcStr1, pcStr2);
	if (n <= 0)
	    break;
	argv[argc] = static_cast < char *>(malloc (strlen (pcStr1) + 1));
	if (!(argv[argc])) {
	    perror ("malloc");
	    while (argc-- > 0)
		free (argv[argc]);
	    free (argv);
	    free (pcStr), free (pcStr1), free (pcStr2);
	    return (-1);
	}
	strcpy (argv[argc++], pcStr1);
	if (n <= 1)
	    break;
	strcpy (pcStr, pcStr2);
    }
    argv[argc] = 0;
    free (pcStr), free (pcStr1), free (pcStr2);

    switch ((pid = fork ())) {
	case 0:		// Child
	    if (p_pcDisplay && *p_pcDisplay)
		putenv (p_pcDisplay);
	    execvp (argv[0], argv);
	    perror (argv[0]);
	    exit (-1);
	case -1:		// Error
	default:		// Parent
	    while (argc-- > 0)
		free (argv[argc]);
	    free (argv);
	    if (pid == -1) {
		perror ("fork()");
		return (-1);
	    }
	    if (p_piPID)
		*p_piPID = pid;
	    waitpid (pid, &iChildStatus, 0);
	    return (iChildStatus);
    }
}				// System()


/**********************************************************************/
void SetWindowSizeConstraint (Widget p_wShell,
			      XtPointer, XEvent * p_pxEvent, Boolean *)
/**********************************************************************/
 /* StructureNotifyMask */
 /* Main window: Non-resizable height, only growing width up to screen
    size */
{
    XConfigureEvent *pxConfigEvt = (XConfigureEvent *) p_pxEvent;

    if (pxConfigEvt->type != ConfigureNotify)
	return;
    XtVaSetValues (p_wShell,
		   XmNminWidth, pxConfigEvt->width,
		   XmNmaxWidth, DisplayWidth (m_pxDisplay,
					      DefaultScreen (m_pxDisplay))
		   /* Should substract 2 x WM-set borders */
		   ,
		   XmNminHeight, pxConfigEvt->height,
		   XmNmaxHeight, pxConfigEvt->height, 0);
    XtRemoveEventHandler (p_wShell, StructureNotifyMask, 0,
			  SetWindowSizeConstraint, 0);
}				// SetWindowSizeConstraint()


/**********************************************************************/
int RefreshXEvents ()
/**********************************************************************/
{
    XEvent          xevt;

    for (; XtAppPending (m_xAppContext);) {
	XtAppNextEvent (m_xAppContext, &xevt);
	XtDispatchEvent (&xevt);
    }
    return (0);
}				// RefreshXEvents()


/**********************************************************************/
int ReverseVideo (Widget p_w)
/**********************************************************************/
{
    Pixel           xForeground,	// 
                    xBackground;

    XtVaGetValues (p_w,
		   XmNbackground, &xBackground,
		   XmNforeground, &xForeground, 0);
    XtVaSetValues (p_w,
		   XmNbackground, xForeground,
		   XmNforeground, xBackground, 0);
    return (0);
}				// ReverseVideo()


/**********************************************************************/
Widget         *WidgetChildren (Widget p_wParent, int *p_piN = 0)
/**********************************************************************/
{
    Widget         *pwChildren;

    XtVaGetValues (p_wParent, XtNchildren, &pwChildren, 0);
    if (p_piN)
	XtVaGetValues (p_wParent, XtNnumChildren, p_piN, 0);
    return (pwChildren);
}				// WidgetChildren()


/**********************************************************************/
int ExtractRCSid (const char *p_pcRCSid, char *p_pcId)
/**********************************************************************/
{
    const size_t    N = strlen (p_pcRCSid) + 1;
    char           *p0 = new char[N],	// 
                   *p1 = new char[N],	// 
                   *p2 = new char[N],	// 
                   *p3 = new char[N],	// 
                   *p4 = new char[N],	// 
                   *p5 = new char[N];
    sscanf (p_pcRCSid, "%s %s %s %s %s %s", p0, p1, p2, p3, p4, p5);
    sprintf (p_pcId, "%s\t%s\t%s\t%s", p1, p2, p3, p5);
    delete[](p0);
    delete[](p1);
    delete[](p2);
    delete[](p3);
    delete[](p4);
    delete[](p5);
    return (0);
}				// ExtractRCSid()


/**********************************************************************/
int SetSimulatorCmd ()
/**********************************************************************/
 /* Set the GUI command line text field according to user's inputs */
{
    char            acCmd[MAXCMDLINESIZE];

    sprintf (acCmd, "%s --fg-root=%s", m_acSimuCmd, m_acFG_Root);
    if (m_iAircraft >= 0) {
	strcat (acCmd, " --aircraft=");
	strcat (acCmd, m_aacAircraftList[m_iAircraft]);
    }
    if (m_iAirport >= 0) {
	strcat (acCmd, " --airport-id=");
	strcat (acCmd, m_aacAirportList[m_iAirport]);
    }
    if (m_fHighVisibility)
	strcat (acCmd, " --disable-clouds --fog-disable");
    if (m_fWindOff)
	strcat (acCmd, " --wind=0@0");
    if (m_oTimeOffset.i) {
	strcat (acCmd, " --time-offset=");
	strcat (acCmd, m_oTimeOffset.ac);
	strcat (acCmd, ":00:00");
    }
    if (m_fUseAtlas) {
	strcat (acCmd, " --nmea=socket,out,0.5,,");
	strcat (acCmd, m_acAtlasPort);
	strcat (acCmd, ",udp");
    }

    XmTextFieldSetString (g_wTF_Cmd, acCmd);
    return (0);
}				// SetSimulatorCmd()


/**********************************************************************/
int SetSensitivity ()
/**********************************************************************/
 /* Set the GUI sensitive states */
{
    XtSetSensitive (g_wMenu_Aircraft, !m_fSimulatorRunning);
    XtSetSensitive (g_wMenu_Airport, !m_fSimulatorRunning);

    XtSetSensitive (g_wF_Options, !m_fSimulatorRunning);

    if (m_fSimulatorRunning) {
	XtMapWidget (g_wPB_Map);
	XtSetSensitive (g_wPB_Map, !m_fAtlasRunning);
    }
    else
	XtUnmapWidget (g_wPB_Map);
    XtVaSetValues (g_wTF_AtlasCmd,
		   XmNeditable, !m_fSimulatorRunning
		   && !m_fAtlasRunning, 0);

    XtSetSensitive (g_wPB_Start, !m_fSimulatorRunning);
    XtVaSetValues (g_wTF_Cmd, XmNeditable, !m_fSimulatorRunning, 0);

    return (0);
}				// SetSensitivity()


/**********************************************************************/
void SetAircraft (Widget p_w, XtPointer p_No, XtPointer)
/**********************************************************************/
 /* XmNactivateCallback */
{
    XmString        XmStr;

    XtVaGetValues (p_w, XmNlabelString, &XmStr, 0);
    XtVaSetValues (g_wL_Aircraft, XmNlabelString, XmStr, 0);

    m_iAircraft = (int) p_No;
    SetSimulatorCmd ();
}				// SetAircraft()


/**********************************************************************/
void SetAirport (Widget p_w, XtPointer p_No, XtPointer)
/**********************************************************************/
 /* XmNactivateCallback */
{
    XmString        XmStr;

    XtVaGetValues (p_w, XmNlabelString, &XmStr, 0);
    XtVaSetValues (g_wL_Airport, XmNlabelString, XmStr, 0);

    m_iAirport = (int) p_No;
    SetSimulatorCmd ();
}				// SetAirport()


/**********************************************************************/
void SetVisibility (Widget, XtPointer, XtPointer p_pvData)
/**********************************************************************/
 /* XmNvalueChangedCallback */
{
    const XmToggleButtonCallbackStruct *poData =
	static_cast < XmToggleButtonCallbackStruct * >(p_pvData);

    m_fHighVisibility = poData->set;
    SetSimulatorCmd ();
}				// SetVisibility()


/**********************************************************************/
void SuppressWind (Widget, XtPointer, XtPointer p_pvData)
/**********************************************************************/
 /* XmNvalueChangedCallback */
{
    const XmToggleButtonCallbackStruct *poData =
	static_cast < XmToggleButtonCallbackStruct * >(p_pvData);

    m_fWindOff = poData->set;
    SetSimulatorCmd ();
}				// SuppressWind()


/**********************************************************************/
void SetAtlasUsage (Widget, XtPointer, XtPointer p_pvData)
/**********************************************************************/
 /* XmNvalueChangedCallback */
{
    const XmToggleButtonCallbackStruct *poData =
	static_cast < XmToggleButtonCallbackStruct * >(p_pvData);
    int             status;

    m_fUseAtlas = poData->set;
    if (m_fUseAtlas)
	XtMapWidget (g_wF_Atlas);
    else {
	XtUnmapWidget (g_wF_Atlas);
	if (m_iAtlasPID) {
	    status = kill (m_iAtlasPID, SIGTERM);
	    if (status == -1)
		perror ("kill <Atlas>");
	    else
		m_iAtlasPID = 0;
	}
    }
    SetSensitivity ();
    SetSimulatorCmd ();
}				// SetAtlasUsage()


/**********************************************************************/
void SetTimeOffset (Widget p_w, XtPointer, XtPointer)
/**********************************************************************/
 /* XmNlosingFocusCallback/XmNactivateCallback */
{
    char           *pc = XmTextFieldGetString (p_w);
    m_oTimeOffset.i = atoi (pc) % 24;
    free (pc);
    if (m_oTimeOffset.i < 0)
	m_oTimeOffset.i += 24;
    if (m_oTimeOffset.i > 12)
	m_oTimeOffset.i -= 24;
    if (m_oTimeOffset.i)
	sprintf (m_oTimeOffset.ac, "%+03d", m_oTimeOffset.i);
    else
	strcpy (m_oTimeOffset.ac, "0");
    XmTextFieldSetString (p_w, m_oTimeOffset.ac);
    SetSimulatorCmd ();
}				// SetTimeOffset()


/**********************************************************************/
int AddSeparator (Widget p_wMenu, const char *p_pcTitle)
/**********************************************************************/
 /* Add a separator in the GUI aircraft/airport menu list */
{
    XmString        XmStr;
    Arg             al[8];
    int             ac;
    Widget          w;

    ac = 0;
    if (!p_pcTitle || !(*p_pcTitle)) {
	XtSetArg (al[ac], XmNorientation, XmHORIZONTAL), ac++;
	XtSetArg (al[ac], XmNseparatorType, XmSHADOW_ETCHED_IN), ac++;
	w = XmCreateSeparator (p_wMenu, (char *) "Separator", al, ac);
	XtManageChild (w);
    }
    else {
	ac = 0;
	XmStr = XmStringGenerate ((char *) p_pcTitle,
				  XmFONTLIST_DEFAULT_TAG, XmCHARSET_TEXT,
				  0);
	XtSetArg (al[ac], XmNlabelString, XmStr), ac++;
	XtSetArg (al[ac], XmNalignment, XmALIGNMENT_CENTER), ac++;
	w = XmCreateLabel (p_wMenu, (char *) "Title", al, ac);
	XmStringFree (XmStr);
	XtManageChild (w);
	ReverseVideo (w);
    }
    return (0);
}				// AddSeparator()


/**********************************************************************/
int AddAircraft (const char *p_pcCode, const char *p_pcName)
/**********************************************************************/
 /* Generates the GUI aircraft menu list */
{
    static int      s_No = 0;
    Widget          w;

    if (strcmp (p_pcCode, SEPARATOR_LABEL) == 0)
	return (AddSeparator (g_wMenu_Aircraft, p_pcName));
    if (s_No > MAX_AIRCRAFTS) {
	fprintf (stderr, "Error: too many aircrafts, "
		 "limit=%d\n", MAX_AIRCRAFTS);
	return (-1);
    }
    if (!(*p_pcName)) {
	fprintf (stderr, "Error: invalid aircraft description, "
		 "aircraft#%d\n", s_No + 1);
	return (-1);
    }
    w = XmCreatePushButton (g_wMenu_Aircraft, (char *) p_pcName, 0, 0);
    XtAddCallback (w, XmNactivateCallback, SetAircraft, (void *) s_No);
    XtManageChild (w);
    strcpy (m_aacAircraftList[s_No], p_pcCode);
    ++s_No;
    return (0);
}				// AddAircraft()


/**********************************************************************/
int AddAirport (const char *p_pcCode, const char *p_pcName)
/**********************************************************************/
 /* Generates the GUI airport menu list */
{
    static int      s_No = 0;
    Widget          w;

    if (strcmp (p_pcCode, SEPARATOR_LABEL) == 0)
	return (AddSeparator (g_wMenu_Airport, p_pcName));
    if (s_No > MAX_AIRPORTS) {
	fprintf (stderr, "Error: too many airports, "
		 "limit=%d\n", MAX_AIRPORTS);
	return (-1);
    }
    if (!(*p_pcName)) {
	fprintf (stderr, "Error: invalid airport description, "
		 "airport#%d\n", s_No + 1);
	return (-1);
    }
    w = XmCreatePushButton (g_wMenu_Airport, (char *) p_pcName, 0, 0);
    XtAddCallback (w, XmNactivateCallback, SetAirport, (void *) s_No);
    XtManageChild (w);
    strcpy (m_aacAirportList[s_No], p_pcCode);
    ++s_No;
    return (0);
}				// AddAirport()


extern          "C" {
    void           *StartAtlas (void *);
}
/**********************************************************************/
    void *StartAtlas (void *)
/**********************************************************************/
 /* Execute the dynamic atlas */
{
    char           *pcCmd;

    pcCmd = XmTextFieldGetString (g_wTF_AtlasCmd);
    System (pcCmd, m_acAtlasXDisplay, &m_iAtlasPID);
    XtFree (pcCmd);
    m_iAtlasPID = 0;
    m_fAtlasRunning = 0;
    SetSensitivity ();
    RefreshXEvents ();
    return (0);
}				// StartAtlas()


extern          "C" {
    void           *StartSimulation (void *);
}
/**********************************************************************/
    void *StartSimulation (void *)
/**********************************************************************/
 /* Execute the flight simulator */
{
    char           *pcCmd;

    pcCmd = XmTextFieldGetString (g_wTF_Cmd);
    System (pcCmd, m_acXDisplay, &m_iSimulatorPID);
    XtFree (pcCmd);
    m_iSimulatorPID = 0;
    m_fSimulatorRunning = 0;
    SetSensitivity ();
    RefreshXEvents ();
    return (0);
}				// StartSimulation()


/**********************************************************************/
void SpawnAtlas (Widget, XtPointer, XtPointer)
/**********************************************************************/
 /* XmNactivateCallback */
 /* Launch the dynamic atlas in a detached threads */
{
    pthread_t       thr;

    if (m_fAtlasRunning)
	return;
    m_fAtlasRunning = 1;
    SetSensitivity ();
    pthread_create (&thr, 0, StartAtlas, 0);
}				// SpawnAtlas()


/**********************************************************************/
void SpawnSimulation (Widget, XtPointer, XtPointer)
/**********************************************************************/
 /* XmNactivateCallback */
 /* Launch the simulation in a detached thread */
{
    pthread_t       thr;

    if (m_fSimulatorRunning)
	return;
    m_fSimulatorRunning = 1;
    SetSensitivity ();
    pthread_create (&thr, 0, StartSimulation, 0);
    if (m_fUseAtlas && !m_fAtlasRunning)
	SpawnAtlas (0, 0, 0);
}				// SpawnSimulation()


/**********************************************************************/
void Exit (Widget, XtPointer fKill, XtPointer)
/**********************************************************************/
 /* XmNokCallback/XmNcancelCallback */
{
    if (fKill) {
	if (m_iSimulatorPID)
	    kill (m_iSimulatorPID, SIGTERM);
	if (m_iAtlasPID)
	    kill (m_iAtlasPID, SIGTERM);
    }
    exit (0);
}				// Exit()


/**********************************************************************/
int SetupGUI (const char *p_pcIniFile)
/**********************************************************************/
 /* Initial setup of the GUI */
{
    FILE           *pF;
    char            acLine[MAXCMDLINESIZE];
    char            acLabel[MAXLABELSIZE];
    char            acP1[MAXCMDLINESIZE];
    char            acP2ext[MAXCMDLINESIZE];
    char            acFormat[16];
    Widget          w;
    int             l;

    memset (m_acAtlasXDisplay, 0, sizeof (m_acAtlasXDisplay));
    pF = fopen (p_pcIniFile, "r");
    if (!pF)
	perror (p_pcIniFile);
    else {
	while ((fgets (acLine, sizeof (acLine), pF))) {
	    memset (acLabel, 0, sizeof (acLabel));
	    memset (acP1, 0, sizeof (acP1));
	    memset (acP2ext, 0, sizeof (acP2ext));
	    sprintf (acFormat, "%%s %%s %%%dc", MAXCMDLINESIZE);
	    (void) sscanf (acLine, acFormat, acLabel, acP1, acP2ext);
	    if ((l = strlen (acP2ext)) > 0)
		acP2ext[l - 1] = 0;
	    if (strcmp (acLabel, AIRCRAFT_LABEL) == 0)
		(void) AddAircraft (acP1, acP2ext);
	    else if (strcmp (acLabel, AIRPORT_LABEL) == 0)
		AddAirport (acP1, acP2ext);
	    else if (strcmp (acLabel, SIMUCMD_LABEL) == 0) {
		sprintf (m_acSimuCmd, "%s %s", acP1, acP2ext);
		if (m_acSimuCmd[(l = strlen (m_acSimuCmd) - 1)] == ' ')
		    m_acSimuCmd[l] = 0;
	    }
	    else if (strcmp (acLabel, FG_ROOT_LABEL) == 0)
		strcpy (m_acFG_Root, acP1);
	    else if (strcmp (acLabel, ATLAS_PORT_LABEL) == 0)
		strcpy (m_acAtlasPort, acP1);
	    else if (strcmp (acLabel, ATLAS_DIR_LABEL) == 0)
		strcpy (m_acAtlasDir, acP1);
	    else if (strcmp (acLabel, ATLAS_XDISPLAY_LABEL) == 0)
		sprintf (m_acAtlasXDisplay, "DISPLAY=%s", acP1);
	    else if (strcmp (acLabel, MAP_CMD_LABEL) == 0) {
		sprintf (m_acMapCmd, "%s %s", acP1, acP2ext);
		if (m_acMapCmd[(l = strlen (m_acMapCmd) - 1)] == ' ')
		    m_acMapCmd[l] = 0;
	    }
	    else
		continue;
	}
	fclose (pF);
    }

    XtAddCallback (g_wTB_HighVisibility, XmNvalueChangedCallback,
		   SetVisibility, 0);

    XtAddCallback (g_wTB_WindOff, XmNvalueChangedCallback, SuppressWind,
		   0);

    XtAddCallback (g_wTB_Atlas, XmNvalueChangedCallback, SetAtlasUsage, 0);

    XtAddCallback (g_wTF_TimeOffset, XmNlosingFocusCallback, SetTimeOffset,
		   0);
    XtAddCallback (g_wTF_TimeOffset, XmNactivateCallback, SetTimeOffset,
		   0);

    XtAddCallback (g_wPB_Start, XmNactivateCallback, SpawnSimulation, 0);
    XtAddCallback (g_wTF_Cmd, XmNactivateCallback, SpawnSimulation, 0);

    XtAddCallback (g_wPB_Map, XmNactivateCallback, SpawnAtlas, 0);
    XtAddCallback (g_wTF_AtlasCmd, XmNactivateCallback, SpawnAtlas, 0);

    w = (WidgetChildren (g_wSh_Exit))[0];	// Exit Message Box
    XtAddCallback (w, XmNokCallback, Exit, (void *) 1);
    XtAddCallback (w, XmNcancelCallback, Exit, 0);

    XmToggleButtonSetState (g_wTB_HighVisibility, !m_fHighVisibility, 0);
    XmToggleButtonSetState (g_wTB_HighVisibility, m_fHighVisibility, 1);

    XmToggleButtonSetState (g_wTB_WindOff, !m_fWindOff, 0);
    XmToggleButtonSetState (g_wTB_WindOff, m_fWindOff, 1);

    XmTextFieldSetString (g_wTF_TimeOffset, m_oTimeOffset.ac);

    sprintf (acLine, "%s --path=%s --udp=%s --fgroot=%s",
	     m_acMapCmd, m_acAtlasDir, m_acAtlasPort, m_acFG_Root);
    XmTextFieldSetString (g_wTF_AtlasCmd, acLine);

    SetSimulatorCmd ();

    return (0);
}				// SetupGUI()


/**********************************************************************/
int SetupHelpPage (const char *p_pcHelpFile)
/**********************************************************************/
 /* Initial setup of the GUI's help page */
{
    char            acTitle[sizeof (RCSid)];
    FILE           *pF;
    char            acLine[128];

    ExtractRCSid (RCSid, acTitle);
    XmTextSetString (g_wT_Help, acTitle);
    pF = fopen (p_pcHelpFile, "r");
    if (!pF) {
	perror (p_pcHelpFile);
	return (-1);
    }
    XmTextInsert (g_wT_Help, XmTextGetLastPosition (g_wT_Help),
		  (char *) "\n" COPYRIGHT "\n");
    while ((fgets (acLine, sizeof (acLine), pF)))
	XmTextInsert (g_wT_Help, XmTextGetLastPosition (g_wT_Help),
		      acLine);
    fclose (pF);
    return (0);
}				// SetupHelpPage()


/**********************************************************************/
void EndApplication (Widget, XtPointer, XtPointer)
/**********************************************************************/
 /* WM_DELETE_WINDOW */
{
    Widget          wMsgBox;

    if (!m_iSimulatorPID && !m_iAtlasPID)
	exit (0);
    wMsgBox = (WidgetChildren (g_wSh_Exit))[0];
    XtManageChild (wMsgBox);
    XtPopup (g_wSh_Exit, XtGrabNone);
}				// Exit()


/**********************************************************************/
int PostSetupGUI ()
/**********************************************************************/
 /* GUI post-setup */
{
    Atom            xWM_DeleteWindow;

    xWM_DeleteWindow = XmInternAtom (m_pxDisplay, "WM_DELETE_WINDOW", 0);
    XmAddWMProtocolCallback (g_wAppSh, xWM_DeleteWindow, EndApplication,
			     0);
    XtVaSetValues (g_wAppSh, XmNdeleteResponse, XmDO_NOTHING, 0);

    XmToggleButtonSetState (g_wTB_Atlas, !m_fUseAtlas, 0);
    XmToggleButtonSetState (g_wTB_Atlas, m_fUseAtlas, 1);

    return (0);
}				// PostSetupGUI()


/**********************************************************************/
int main (int argc, char **argv)
/**********************************************************************/
{
    const char     *pcIniFile = DEFAULT_INI_FILENAME,	// 
	*pcHlpFile = DEFAULT_HLP_FILENAME;
    char           *pc;
    int             fError = 0,	// 
	fUsage = 0,		// 
	status;

    /* Connect to the X server */
    XtSetLanguageProc (0, 0, 0);
    XtToolkitInitialize ();
    status = XtToolkitThreadInitialize ();
    if (!status) {
	perror ("XtToolkitThreadInitialize");
	exit (~0);
    }
    m_xAppContext = XtCreateApplicationContext ();
    m_pxDisplay = XtOpenDisplay (m_xAppContext, 0, argv[0], APPNAME,
				 0, 0, &argc, argv);
    if (!m_pxDisplay) {
	fprintf (stderr, "%s: can't open display, exiting...\n", argv[0]);
	exit (~0);
    }

    /* Get non-X-specific command line parameters */
    for (char c; (c = getopt (argc, argv, "a:f:hV")) != EOF;)
	switch (c) {
	    case 'f':		// Initialization file
		pcIniFile = optarg;
		break;
	    case 'a':		// Help file
		pcHlpFile = optarg;
		break;
	    case 'h':		// Help
	    case 'V':		// Version
		fUsage = 1;
		break;
	    case '?':
	    default:
		fError = 1;
	}
    fError = (fError || (argc > optind));	// Too many parameters?
    fUsage = (fUsage || fError);
    if (fUsage) {
	FILE           *pFOut = (fError ? stderr : stdout);
	fprintf (stdout, "%s\n", RCSid + 5);
	fprintf (stdout,
		 "Flight simulator - "
		 "Set of aircrafts and airports indicated in ini. file\n");
	fprintf (pFOut,
		 "Usage:\t%s [-toolkitoption...] [-f IniFile] [-a HelpFile]\n",
		 argv[0]);
	fprintf (pFOut, "\t%s -h | -V\n", argv[0]);
	fprintf (pFOut, "\t\t-i IniFile"
		 "\tUse the indicated initialization file\n"
		 "\t\t\t\t(default=\"%s\")\n", DEFAULT_INI_FILENAME);
	fprintf (pFOut, "\t\t-a HelpFile"
		 "\tHelp content to display when requested\n"
		 "\t\t\t\t(default=\"%s\")\n", DEFAULT_HLP_FILENAME);
	fprintf (pFOut, "\t\t-h\t\tOutput this help and exit\n");
	fprintf (pFOut, "\t\t-V\t\tOutput version information and exit\n");
	return (fError ? ~0 : 0);
    }

    /* General initialization */
    m_iAircraft = -1;
    m_iAirport = -1;
    m_fSimulatorRunning = 0;
    m_fAtlasRunning = 0;
    m_iSimulatorPID = 0;
    m_iAtlasPID = 0;
    sprintf (m_acXDisplay, "DISPLAY=%s",
	     ((pc = getenv ("DISPLAY")) ? pc : ":0.0"));

    /* This converter is not registered internally by Motif */
    XmRepTypeInstallTearOffModelConverter ();

    /* Create the GUI */
    create_g_wAppSh (m_pxDisplay, (char *) "Flight Simulator", argc, argv);
    XtAddEventHandler (g_wAppSh, StructureNotifyMask, 0,
		       SetWindowSizeConstraint, 0);
    create_g_wSh_Help (g_wAppSh);
    SetupHelpPage (pcHlpFile);
    create_g_wSh_Exit (g_wAppSh);
    status = SetupGUI (pcIniFile);
    if (status == -1)
	return (~0);
    XtRealizeWidget (g_wAppSh);
    PostSetupGUI ();

    /* Entering infinite X event processing loop... */
    XtAppMainLoop (m_xAppContext);

    return (0);
}				// main()


/* 
 *$Log: flightsimu.C,v $
 *Revision 1.18  2004/08/15 11:12:45  RogerSeguin
 *Display author's contact details
 *
 *
 */
