/*
 * Copyright (c) 2002, 2004 Roger Seguin <roger_seguin@msn.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

static char     rcsid[] =
    "$Id: displayrgb.C,v 1.4 2004/08/15 11:39:24 RogerSeguin Exp $";

#define COPYRIGHT	"Copyright (c) 2002, 2004 Roger Seguin "	\
    			"<roger_seguin@msn.com, http://rmlx.dyndns.org>"

/*
Description:
	Display colors listed in /usr/X11R6/lib/X11/rgb.txt
	Setup e.g.:
		Add
			DisplayRGB*XmToggleButton.indicatorSize: 40
		in ~/.Xdefaults
	Usage:
		displayrgb
		displayrgb -h
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <memory.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/RepType.h>
#include <Xm/Command.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/SelectioB.h>
#include <Xm/Text.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/ToggleB.h>


#include "rgbgui.h"


#define APPNAME	"DisplayRGB"


namespace {
    XtAppContext    m_xAppContext;
    int             m_fVerbose;
}


/**********************************************************************/
void CallbackResetColor (Widget p_w, XtPointer, XtPointer p_pvData)
/**********************************************************************/
 /* XmNvalueChangedCallback */
{
    const XmToggleButtonCallbackStruct *poData =
	static_cast < XmToggleButtonCallbackStruct * >(p_pvData);

    if (!(poData->set))
	XmToggleButtonSetState (p_w, 1, 0);
}				// CallbackResetColor()


/**********************************************************************/
int Setup (const char *p_pcRGB_TxtFile)
/**********************************************************************/
{
    const int       iRootWinDepth =
	DefaultDepthOfScreen (DefaultScreenOfDisplay (XtDisplay (wAppSh)));
    FILE           *pF;
    int             r, g, b;
    int             iPrevR = -1, iPrevG = -1, iPrevB = -1;
    char            acLine[128];
    char            acColorName[64],	// 
                    acDummy[4];
    Widget          wTB;
    Arg             axArg[8];
    XrmValue        oCol1, oCol2;
    int             ArgCount;
    int             lineno = 0,	// 
	ncolors = 0;
    int             fError;
    int             status;

    if (iRootWinDepth == -1) {
	fprintf (stderr, " ** Error: DefaultScreenOfDisplay()\n");
	return (-1);
    }
    pF = fopen (p_pcRGB_TxtFile, "r");
    if (!pF) {
	perror (p_pcRGB_TxtFile);
	return (-1);
    }
    while ((fgets (acLine, sizeof (acLine), pF))) {
	++lineno;
	status = sscanf (acLine, "%d%d%d%63s%1s",
			 &r, &g, &b, acColorName, acDummy);
	if (status != 4)
	    /* Comment or color name is the one containing spaces */
	    continue;
	if ((r == iPrevR) && (g == iPrevG) && (b == iPrevB))
	    /* Different names for the same color (e.g. Gray/Grey) */
	    continue;
	iPrevR = r, iPrevG = g, iPrevB = b;
	ArgCount = 0;
	XtSetArg (axArg[ArgCount], XmNvisibleWhenOff, 0), ArgCount++;
	XtSetArg (axArg[ArgCount], XmNfillOnSelect, 1), ArgCount++;
	fError = 0;
	oCol1.addr = acColorName;
	oCol1.size = strlen (oCol1.addr) + 1;
	oCol2.addr = 0;
	XtConvertAndStore (wRC, XmRString, &oCol1, XmRPixel, &oCol2);
	if (oCol2.addr)
	    XtSetArg (axArg[ArgCount],
		      XmNselectColor, (*((Pixel *) oCol2.addr))),
		ArgCount++;
	else {
	    fprintf (stderr, " ** Error: %s\n", acColorName);
	    fError = 1;
	}
	XtSetArg (axArg[ArgCount], XmNindicatorOn, !fError), ArgCount++;
	XtSetArg (axArg[ArgCount], XmNset, !fError), ArgCount++;
	sprintf (acLine, "%4d: %s", lineno, acColorName);
	wTB = XmCreateToggleButton (wRC, (char *) acLine, axArg, ArgCount);
	XtAddCallback (wTB, XmNvalueChangedCallback, CallbackResetColor,
		       0);
	XtManageChild (wTB);
	++ncolors;
    }
    fclose (pF);
    if (m_fVerbose)
	fprintf (stdout, "%s: %d lines, %d colors.\n",
		 p_pcRGB_TxtFile, lineno, ncolors);
    return (0);
}				// Setup()


/**********************************************************************/
int main (int argc, char **argv)
/**********************************************************************/
{
    const char      acDefaultRGB_TxtFile[] = "/usr/X11R6/lib/X11/rgb.txt",	// 
	*pcRGB_TxtFile = acDefaultRGB_TxtFile;
    Display        *pxDisplay;
    char            acTitle[sizeof (rcsid)],	// 
                   *pcTitle;
    const char     *pcId;
    int             fError = 0,	// 
	fUsage = 0,		// 
	status;

    /* Enable Localisation of the Application */
    XtSetLanguageProc (0, 0, 0);

    /* Initialise the X Toolkit */
    XtToolkitInitialize ();

    /* Create a Global Application Context */
    m_xAppContext = XtCreateApplicationContext ();

    /* Open the Display */
    pxDisplay = XtOpenDisplay (m_xAppContext, 0, argv[0], APPNAME,
			       0, 0, &argc, argv);
    if (!pxDisplay) {
	fprintf (stderr, "%s: can't open display, exiting...\n", argv[0]);
	exit (~0);
    }

    /* Get non-X-specific command line parameters */
    m_fVerbose = 0;
    for (char c; (c = getopt (argc, argv, "hvV")) != EOF;)
	switch (c) {
	    case 'h':		// Help
	    case 'V':		// Version
		fUsage = 1;
		break;
	    case 'v':		// Verbose
		m_fVerbose = 1;
		break;
	    case '?':
	    default:
		fError = 1;
	}
    fError = (fError || (argc > optind + 1));	// Too many parameters?
    fUsage = (fUsage || fError);
    if (fUsage) {
	FILE           *pFOut = (fError ? stderr : stdout);
	fprintf (stdout, "%s\n", rcsid + 5);
	fprintf (stdout, COPYRIGHT "\n");
	fprintf (stdout,
		 "Read RGB text file and display associated color palette\n");
	fprintf (pFOut,
		 "Usage:\t%s [-toolkitoption...] [-v] [RGB_TxtFile]\n",
		 argv[0]);
	fprintf (pFOut, "\t%s -h | -V\n", argv[0]);
	fprintf (pFOut,
		 "\t\t-v\tVerbose:\tprint messages while executing\n");
	fprintf (pFOut, "\t\t-h\tHelp:\t\toutput this help and exit\n");
	fprintf (pFOut,
		 "\t\t-V\tVersion:\toutput version information and exit\n");
	return (fError ? ~0 : 0);
    }
    if (optind < argc)
	pcRGB_TxtFile = argv[optind];

    /* This converter is not registered internally by Motif */
    XmRepTypeInstallTearOffModelConverter ();

    /* Set up the GUI */
    for (pcId = rcsid + 4, pcTitle = acTitle;
	 ((*pcTitle = *pcId)); pcTitle++, pcId++)
	if (*pcTitle == '\t')
	    *pcTitle = ' ';
    create_wAppSh (pxDisplay, acTitle, argc, argv);
    status = Setup (pcRGB_TxtFile);
    if (status == -1)
	return (~0);
    XtRealizeWidget (wAppSh);

    /* Entering X Main Loop... */
    XtAppMainLoop (m_xAppContext);

    /* NOTREACHED */
    return (0);
}				// main()


/*
$Log: displayrgb.C,v $
Revision 1.4  2004/08/15 11:39:24  RogerSeguin
Display author's contact details

Revision 1.1  2002/03/21 19:05:19  RogerSeguin
Initial revision

*/
